---
title: "Créer sa première image avec un Dockerfile"
course: "docker"
slug: "5-creer-sa-premiere-image-avec-un-dockerfile"
order: 5
createdAt: 2025-09-24
updatedAt: 2025-09-24
---

Créer une image Docker personnalisée commence par un fichier spécial : le **Dockerfile**.

---

## Qu'est-ce qu'un Dockerfile ?

Un **Dockerfile** est un simple fichier texte contenant une liste d'instructions que Docker va suivre pour construire une image. Chaque instruction crée une **couche** (layer) dans l'image finale.

> [!info] Concept clé
> Le Dockerfile est comme une recette : il décrit étape par étape comment préparer votre "plat" (l'image).

---

## Exemple simple : application Python

Imaginons qu'on ait une app Python qui contient deux fichiers :

**main.py :**

```python
print("Bonjour depuis Docker !")
```

**requirements.txt :**

```txt
requests==2.31.0
```

---

## Dockerfile basique

```dockerfile
# Utiliser une image de base officielle
FROM python:3.12-slim

# Définir le répertoire de travail
WORKDIR /app

# Copier les dépendances et les installer
COPY requirements.txt .
RUN pip install --no-cache-dir -r requirements.txt

# Copier le code de l'application
COPY . .

# Commande exécutée au démarrage du conteneur
CMD ["python", "main.py"]
```

---

## Instructions Dockerfile essentielles

### FROM

Définit l'image de base sur laquelle construire.

```dockerfile
FROM python:3.12-slim
```

> [!tip] Bonne pratique
> Toujours spécifier une version précise, jamais `latest` !

### WORKDIR

Définit le répertoire de travail dans l'image.

```dockerfile
WORKDIR /app
```

### COPY

Copie des fichiers de l'hôte vers l'image.

```dockerfile
COPY requirements.txt .
COPY src/ ./src/
```

### RUN

Exécute une commande pendant la construction de l'image.

```dockerfile
RUN pip install --no-cache-dir -r requirements.txt
RUN apt-get update && apt-get install -y curl
```

### CMD

Définit la commande par défaut exécutée au lancement du conteneur.

```dockerfile
CMD ["python", "main.py"]
```

### EXPOSE

Indique quel port l'application écoute.

```dockerfile
EXPOSE 8000
```

---

## Construire l'image

Dans le terminal, se placer dans le dossier contenant le Dockerfile et exécuter :

```bash
docker build -t mon-app-python .
```

**Explication :**

- `-t mon-app-python` : nomme l'image
- `.` : utilise le répertoire courant comme contexte de build

---

## Lancer un conteneur basé sur cette image

```bash
docker run --rm mon-app-python
```

**Options :**

- `--rm` : supprime le conteneur après son exécution

---

## Structure du projet

```
mon-projet/
├── Dockerfile
├── requirements.txt
└── main.py
```

---

## Dockerfile optimisé (bonnes pratiques)

```dockerfile
# Image de base légère et spécifique
FROM python:3.12-slim

# Créer un utilisateur non-root pour la sécurité
RUN adduser --disabled-password --gecos '' --shell /bin/bash appuser

# Définir le répertoire de travail
WORKDIR /app

# Copier et installer les dépendances en premier (optimisation cache)
COPY requirements.txt .
RUN pip install --no-cache-dir -r requirements.txt

# Copier le code source
COPY . .

# Changer de propriétaire des fichiers
RUN chown -R appuser:appuser /app

# Passer à l'utilisateur non-root
USER appuser

# Port sur lequel l'application écoute
EXPOSE 8000

# Commande de démarrage
CMD ["python", "main.py"]
```

---

## Optimisations importantes

### 1. Ordre des instructions

Placer les instructions qui changent peu **en premier** pour optimiser le cache Docker.

**Bon :**

```dockerfile
COPY requirements.txt .
RUN pip install -r requirements.txt
COPY . .
```

**Moins bon :**

```dockerfile
COPY . .
RUN pip install -r requirements.txt
```

### 2. Combiner les commandes RUN

**Bon :**

```dockerfile
RUN apt-get update && apt-get install -y \
    curl \
    vim \
    && rm -rf /var/lib/apt/lists/*
```

**Moins bon :**

```dockerfile
RUN apt-get update
RUN apt-get install -y curl
RUN apt-get install -y vim
```

### 3. Utiliser .dockerignore

Créer un fichier `.dockerignore` pour exclure les fichiers inutiles :

```
.git
.gitignore
README.md
__pycache__
*.pyc
.env
```

### 4. Sécurité

- Utiliser un utilisateur non-root
- Spécifier des versions précises
- Nettoyer les caches après installation

---

## Exemple avec application web

```dockerfile
FROM python:3.12-slim

# Variables d'environnement
ENV PYTHONUNBUFFERED=1
ENV PYTHONDONTWRITEBYTECODE=1

# Créer utilisateur
RUN adduser --disabled-password --gecos '' appuser

WORKDIR /app

# Installer les dépendances système si nécessaire
RUN apt-get update && apt-get install -y --no-install-recommends \
    build-essential \
    && rm -rf /var/lib/apt/lists/*

# Copier et installer les dépendances Python
COPY requirements.txt .
RUN pip install --no-cache-dir -r requirements.txt

# Copier le code
COPY . .
RUN chown -R appuser:appuser /app

USER appuser

EXPOSE 8000

# Vérification de santé
HEALTHCHECK --interval=30s --timeout=3s --start-period=5s --retries=3 \
  CMD curl -f http://localhost:8000/health || exit 1

CMD ["python", "-m", "uvicorn", "app:app", "--host", "0.0.0.0", "--port", "8000"]
```

---

## Commandes utiles

### Voir l'historique des couches

```bash
docker history mon-app-python
```

### Inspecter l'image

```bash
docker inspect mon-app-python
```

### Build sans cache

```bash
docker build --no-cache -t mon-app-python .
```

---

## À retenir

- **FROM** : image de base
- **WORKDIR** : répertoire de travail
- **COPY** : copie des fichiers
- **RUN** : exécute des commandes à la construction
- **CMD** : commande par défaut au lancement
- **Ordre des instructions** : important pour l'optimisation du cache
- **Utilisateur non-root** : essentiel pour la sécurité
- **Versions spécifiques** : éviter `latest`

---

## Liens vers la partie suivante

- [[6. Gestion des données avec les volumes]]

---

## Tags

#docker #dockerfile #build #images #python #bonnes-pratiques #sécurité